<?php
error_reporting(E_ALL);
session_start();
include('assets/inc/config.php');

// Get patient ID from URL
$patient_id = isset($_GET['pat_id']) ? $_GET['pat_id'] : null;

// Fetch patient details
$patient = null;
if ($patient_id) {
    $query = "SELECT * FROM his_patients WHERE pat_id = ?";
    $stmt = $mysqli->prepare($query);
    $stmt->bind_param('i', $patient_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $patient = $result->fetch_assoc();
}

// Handle form submission for adding medicine orders
if(isset($_POST['add_medicine_order'])) {
    $pat_id = $_POST['pat_id'];
    $department = $_POST['department'];
    $patient_number = $_POST['patient_number'];
    
    // Process multiple medicine orders
    if(isset($_POST['medicine_name']) && is_array($_POST['medicine_name'])) {
        foreach($_POST['medicine_name'] as $index => $name) {
            if(!empty($name)) {
                $description = $_POST['medicine_description'][$index];
                $qty = $_POST['medicine_qty'][$index];
                $price = $_POST['medicine_price'][$index];
                
                // Calculate total price
                $total_price = $qty * $price;
                
                // Insert into database
                $query = "INSERT INTO his_medicine_orders (pat_id, department, patient_number, name, description, qty, price, total_price) 
                          VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
                $stmt = $mysqli->prepare($query);
                $stmt->bind_param('issssidd', $pat_id, $department, $patient_number, $name, $description, $qty, $price, $total_price);
                if($stmt->execute()) {
                    // Success for this item
                } else {
                    $err = "Error recording medicine order: " . $mysqli->error;
                }
            }
        }
        
        if (!isset($err)) {
            $success = "Medicine orders added successfully";
        }
    } else {
        $err = "Please add at least one medicine";
    }
}

// Valid payment methods matching DB ENUM
$valid_payment_methods = ['Cash', 'Credit Card', 'Debit Card', 'Bank Transfer', 'Insurance'];

// Handle form submission for adding payment
if(isset($_POST['add_payment'])) {
    $pat_id = $_POST['pat_id'];
    $total_amount_due = floatval($_POST['total_amount_due']);
    $paid_amount = floatval($_POST['paid_amount']);
    $discount_amount = floatval($_POST['discount_amount']);
    $payment_method = trim($_POST['payment_method']);
    $remarks = trim($_POST['remarks']);
    
    // Validate payment method
    if (!in_array($payment_method, $valid_payment_methods)) {
        $err = "Invalid payment method selected. Please choose a valid option.";
    } elseif ($paid_amount <= 0 || $paid_amount > ($total_amount_due - $discount_amount)) {
        $err = "Paid amount must be greater than 0 and not exceed the due amount after discount.";
    } else {
        // Generate payment reference
        $payment_ref = 'PAY' . date('Ymd') . rand(1000, 9999);
        
        // Create bill first
        $query = "INSERT INTO his_bills (bill_number, pat_id, bill_date, description, amount, status) 
                  VALUES (?, ?, CURDATE(), 'Medicine Orders Payment', ?, 'Partial')";
        $stmt = $mysqli->prepare($query);
        $bill_amount = $total_amount_due - $discount_amount;
        $stmt->bind_param('sid', $payment_ref, $pat_id, $bill_amount);
        
        if($stmt->execute()) {
            $bill_id = $mysqli->insert_id;
            
            // Insert payment record
            $query = "INSERT INTO his_payments (payment_ref, pat_id, bill_id, payment_date, amount, payment_method, notes) 
                      VALUES (?, ?, ?, CURDATE(), ?, ?, ?)";
            $stmt = $mysqli->prepare($query);
            $stmt->bind_param('siidss', $payment_ref, $pat_id, $bill_id, $paid_amount, $payment_method, $remarks);
            
            if($stmt->execute()) {
                $success = "Payment recorded successfully. Reference: $payment_ref. Remaining due: ₹" . number_format($total_amount_due - $discount_amount - $paid_amount, 2);
            } else {
                $err = "Error recording payment: " . $mysqli->error;
                // Rollback bill if payment fails
                $rollback_query = "DELETE FROM his_bills WHERE bill_id = ?";
                $rollback_stmt = $mysqli->prepare($rollback_query);
                $rollback_stmt->bind_param('i', $bill_id);
                $rollback_stmt->execute();
            }
        } else {
            $err = "Error creating bill: " . $mysqli->error;
        }
    }
}

// Fetch medicine orders for this patient
$medicine_orders = [];
if ($patient_id) {
    $query = "SELECT * FROM his_medicine_orders WHERE pat_id = ?";
    $stmt = $mysqli->prepare($query);
    $stmt->bind_param('i', $patient_id);
    $stmt->execute();
    $result = $stmt->get_result();
    while ($row = $result->fetch_assoc()) {
        $medicine_orders[] = $row;
    }
}

// Fetch payment history with join to bills for better display
$payments = [];
if ($patient_id) {
    $query = "SELECT p.*, b.amount as bill_amount FROM his_payments p 
              LEFT JOIN his_bills b ON p.bill_id = b.bill_id 
              WHERE p.pat_id = ? ORDER BY p.payment_date DESC";
    $stmt = $mysqli->prepare($query);
    $stmt->bind_param('i', $patient_id);
    $stmt->execute();
    $result = $stmt->get_result();
    while ($row = $result->fetch_assoc()) {
        $payments[] = $row;
    }
}

// Calculate totals
$total_qty = 0;
$total_price = 0;
foreach ($medicine_orders as $order) {
    $total_qty += intval($order['qty']);
    $total_price += floatval($order['total_price']);
}

// Calculate payment totals
$total_paid = 0;
foreach ($payments as $payment) {
    $total_paid += floatval($payment['amount']);
}
$remaining_due = max(0, $total_price - $total_paid); // Ensure non-negative
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <!--Head-->
    <?php include('assets/inc/head.php');?>
    <style>
        :root {
            --bg-primary: #ffffff;
            --bg-secondary: #f8fbff;
            --text-primary: #1a1a1a;
            --text-secondary: #666666;
            --border-color: #e5e7eb;
            --shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.1), 0 1px 2px 0 rgba(0, 0, 0, 0.06);
            --shadow-lg: 0 10px 15px -3px rgba(0, 0, 0, 0.1), 0 4px 6px -2px rgba(0, 0, 0, 0.05);
            --transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1);
            --radius: 8px;
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            background: linear-gradient(135deg, var(--bg-primary) 0%, #f0f9ff 100%);
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
            line-height: 1.6;
            color: var(--text-primary);
            min-height: 100vh;
        }

        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 20px;
            width: 100%; /* col-12 equivalent */
        }

        .section {
            background: var(--bg-primary);
            border-radius: var(--radius);
            box-shadow: var(--shadow);
            margin-bottom: 24px;
            overflow: hidden;
            border: 1px solid var(--border-color);
            transition: var(--transition);
            width: 100%; /* col-12 equivalent */
        }

        .section:hover {
            box-shadow: var(--shadow-lg);
            transform: translateY(-1px);
        }

        .section-header {
            background: linear-gradient(135deg, #e0f2fe 0%, #f0f9ff 100%);
            padding: 20px;
            border-bottom: 1px solid var(--border-color);
            font-size: 1.25rem;
            font-weight: 600;
            color: var(--text-primary);
        }

        .section-content {
            padding: 24px;
        }

        .patient-header {
            background: linear-gradient(135deg, #e0f2fe 0%, #f0f9ff 100%);
            border-radius: var(--radius);
            padding: 24px;
            margin-bottom: 24px;
            box-shadow: var(--shadow);
            border: 1px solid var(--border-color);
            width: 100%; /* col-12 equivalent */
        }

        .patient-header h2 {
            margin-bottom: 16px;
            color: var(--text-primary);
            font-size: 1.5rem;
        }

        .patient-info {
            display: flex;
            flex-direction: column;
            gap: 12px;
            font-size: 0.95rem;
        }

        .patient-info-item {
            display: flex;
            align-items: center;
            gap: 8px;
        }

        .form-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 16px;
            margin-bottom: 20px;
        }

        .form-group {
            display: flex;
            flex-direction: column;
        }

        .form-group label {
            font-weight: 500;
            margin-bottom: 6px;
            color: var(--text-secondary);
            font-size: 0.9rem;
        }

        .form-control {
            padding: 12px 16px;
            border: 1px solid var(--border-color);
            border-radius: var(--radius);
            font-size: 1rem;
            transition: var(--transition);
            background: var(--bg-primary);
        }

        .form-control:focus {
            outline: none;
            border-color: #3b82f6;
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
        }

        .btn {
            padding: 12px 24px;
            border: none;
            border-radius: var(--radius);
            font-weight: 500;
            font-size: 0.95rem;
            cursor: pointer;
            transition: var(--transition);
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 8px;
        }

        .btn-primary {
            background: #3b82f6;
            color: white;
        }

        .btn-primary:hover {
            background: #2563eb;
            transform: translateY(-1px);
        }

        .btn-secondary {
            background: #6b7280;
            color: white;
        }

        .btn-secondary:hover {
            background: #4b5563;
        }

        .btn-danger {
            background: #ef4444;
            color: white;
        }

        .btn-danger:hover {
            background: #dc2626;
        }

        .btn-add {
            background: #10b981;
            color: white;
            padding: 8px 16px;
            font-size: 0.9rem;
            margin-bottom: 16px;
        }

        .btn-add:hover {
            background: #059669;
        }

        .table {
            width: 100%;
            border-collapse: collapse;
            margin: 16px 0;
            font-size: 0.9rem;
        }

        .table th,
        .table td {
            padding: 12px 8px;
            text-align: left;
            border-bottom: 1px solid var(--border-color);
        }

        .table th {
            background: #f8fafc;
            font-weight: 600;
            color: var(--text-secondary);
            font-size: 0.85rem;
            text-transform: uppercase;
            letter-spacing: 0.05em;
        }

        .table tr:hover {
            background: #f8fafc;
        }

        .table .total-row {
            background: #f0f9ff;
            font-weight: 600;
            border-top: 2px solid var(--border-color);
        }

        .medicine-row {
            display: flex;
            flex-direction: column;
            gap: 12px;
            padding: 16px;
            border: 1px solid var(--border-color);
            border-radius: var(--radius);
            background: #fafbfc;
            margin-bottom: 16px;
        }

        .medicine-row:last-child {
            margin-bottom: 0;
        }

        .medicine-fields {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 12px;
        }

        .payment-summary {
            background: #f0f9ff;
            padding: 16px;
            border-radius: var(--radius);
            border: 1px solid var(--border-color);
            margin: 20px 0;
        }

        .payment-summary h4 {
            margin-bottom: 12px;
            color: var(--text-primary);
        }

        .payment-row {
            display: flex;
            justify-content: space-between;
            padding: 8px 0;
            border-bottom: 1px solid var(--border-color);
        }

        .payment-row:last-child {
            border-bottom: none;
            font-weight: 600;
            color: #059669;
        }

        .alert {
            padding: 12px 16px;
            border-radius: var(--radius);
            margin-bottom: 20px;
            font-weight: 500;
        }

        .alert-success {
            background: #f0fdf4;
            color: #166534;
            border: 1px solid #bbf7d0;
        }

        .alert-danger {
            background: #fef2f2;
            color: #dc2626;
            border: 1px solid #fecaca;
        }

        /* Payment Modal Styles */
        .payment-modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.7);
            z-index: 1000;
            justify-content: center;
            align-items: center;
            overflow-y: auto;
            padding: 20px;
        }

        .payment-modal.active {
            display: flex;
        }

        .payment-modal-content {
            background: white;
            border-radius: var(--radius);
            box-shadow: var(--shadow-lg);
            width: 100%;
            max-width: 600px;
            max-height: 90vh;
            overflow-y: auto;
            position: relative;
        }

        .payment-modal-header {
            background: linear-gradient(135deg, #e0f2fe 0%, #f0f9ff 100%);
            padding: 20px;
            border-bottom: 1px solid var(--border-color);
            font-size: 1.25rem;
            font-weight: 600;
            color: var(--text-primary);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .payment-modal-close {
            background: none;
            border: none;
            font-size: 1.5rem;
            cursor: pointer;
            color: var(--text-secondary);
        }

        .payment-modal-body {
            padding: 24px;
        }

        @media (max-width: 768px) {
            .container {
                padding: 12px;
            }
            
            .section-content {
                padding: 16px;
            }
            
            .medicine-fields {
                grid-template-columns: 1fr;
            }
            
            .payment-row {
                flex-direction: column;
                gap: 4px;
            }
            
            .form-grid {
                grid-template-columns: 1fr;
            }
            
            .payment-modal-content {
                width: 95%;
                margin: 20px auto;
            }
        }
    </style>
</head>
<body>
    <div id="wrapper">
        <!-- Topbar Start -->
        <?php include("assets/inc/nav.php");?>
        <!-- end Topbar -->

        <!-- ========== Left Sidebar Start -->
        <?php include("assets/inc/sidebar.php");?>
        <!-- Left Sidebar End -->

        <div class="content-page">
            <div class="content">
                <div class="container-fluid">
                    <div class="container">
                        <!-- Display success/error messages -->
                        <?php if(isset($success)) { ?>
                        <div class="alert alert-success">
                            <strong>Success!</strong> <?php echo $success; ?>
                        </div>
                        <?php } ?>
                        
                        <?php if(isset($err)) { ?>
                        <div class="alert alert-danger">
                            <strong>Error!</strong> <?php echo $err; ?>
                        </div>
                        <?php } ?>
                        
                        <?php if($patient): ?>
                        <!-- Patient Header -->
                        <div class="patient-header">
                            <h2>Patient Dashboard - <?php echo htmlspecialchars($patient['title'] . ' ' . $patient['pat_fname'] . ' ' . $patient['pat_lname']); ?></h2>
                            <div class="patient-info">
                                <div class="patient-info-item">
                                    <span>👤</span>
                                    <span><strong>Name:</strong> <?php echo htmlspecialchars($patient['title'] . ' ' . $patient['pat_fname'] . ' ' . $patient['pat_lname']); ?></span>
                                </div>
                                <div class="patient-info-item">
                                    <span>🆔</span>
                                    <span><strong>Patient ID:</strong> <?php echo htmlspecialchars($patient['pat_number']); ?></span>
                                </div>
                                <div class="patient-info-item">
                                    <span>🏥</span>
                                    <span><strong>Department:</strong> <?php echo htmlspecialchars($patient['dept']); ?></span>
                                </div>
                                <div class="patient-info-item">
                                    <span>📅</span>
                                    <span><strong>Registration Date:</strong> <?php echo date('Y-m-d H:i:s', strtotime($patient['pat_date_joined'])); ?></span>
                                </div>
                                <div class="patient-info-item">
                                    <span>🏷️</span>
                                    <span><strong>Type:</strong> <?php echo htmlspecialchars($patient['pat_type']); ?></span>
                                </div>
                            </div>
                        </div>
  <button type="button" class="btn btn-primary" id="open-payment-modal">
                                    💳 Record New Payment
                                </button>

                                <!-- Payment History -->
                                <?php if(count($payments) > 0): ?>
                                <div style="margin-top: 32px;">
                                    <h4 style="margin-bottom: 16px;">Payment History</h4>
                                    <div class="table-responsive">
                                        <table class="table">
                                            <thead>
                                                <tr>
                                                    <th>#</th>
                                                    <th>Reference</th>
                                                    <th>Method</th>
                                                    <th>Date</th>
                                                    <th>Amount</th>
                                                    <th>Bill Amount</th>
                                                    <th>Remarks</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php foreach($payments as $index => $payment): ?>
                                                <tr>
                                                    <td><?php echo $index + 1; ?></td>
                                                    <td><?php echo htmlspecialchars($payment['payment_ref']); ?></td>
                                                    <td><?php echo htmlspecialchars($payment['payment_method'] ?: 'N/A'); ?></td>
                                                    <td><?php echo date('Y-m-d H:i', strtotime($payment['payment_date'])); ?></td>
                                                    <td>₹<?php echo number_format($payment['amount'], 2); ?></td>
                                                    <td>₹<?php echo number_format($payment['bill_amount'], 2); ?></td>
                                                    <td><?php echo htmlspecialchars($payment['notes'] ?? ''); ?></td>
                                                </tr>
                                                <?php endforeach; ?>
                                            </tbody>
                                        </table>
                                    </div>
                                </div>
                                <?php else: ?>
                                <div style="margin-top: 24px; padding: 24px; text-align: center; color: var(--text-secondary);">
                                    No payment history found for this patient.
                                </div>
                                <?php endif; ?>
                            </div>
                        </div>

                        <!-- Medicine Orders Section -->
                        <div class="section">
                            <div class="section-header">💊 Medicine Orders</div>
                            <div class="section-content">
                                <h4 style="margin-bottom: 16px;">Medicine Order History</h4>
                                
                                <?php if(count($medicine_orders) > 0): ?>
                                <div class="table-responsive">
                                    <table class="table">
                                        <thead>
                                            <tr>
                                                <th>#</th>
                                                <th>Medicine</th>
                                                <th>Description</th>
                                                <th>Qty</th>
                                                <th>Price</th>
                                                <th>Total</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach($medicine_orders as $index => $order): ?>
                                            <tr>
                                                <td><?php echo $index + 1; ?></td>
                                                <td><?php echo htmlspecialchars($order['name']); ?></td>
                                                <td><?php echo htmlspecialchars($order['description']); ?></td>
                                                <td><?php echo htmlspecialchars($order['qty']); ?></td>
                                                <td>₹<?php echo number_format($order['price'], 2); ?></td>
                                                <td>₹<?php echo number_format($order['total_price'], 2); ?></td>
                                            </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                        <tfoot>
                                            <tr class="total-row">
                                                <td colspan="3">Total</td>
                                                <td><?php echo $total_qty; ?></td>
                                                <td></td>
                                                <td>₹<?php echo number_format($total_price, 2); ?></td>
                                            </tr>
                                        </tfoot>
                                    </table>
                                </div>
                                <?php else: ?>
                                <p style="text-align: center; color: var(--text-secondary); padding: 40px 20px;">No medicine orders found for this patient.</p>
                                <?php endif; ?>
                            </div>
                        </div>

                        <!-- Payment Section -->
                        <div class="section">
                            <div class="section-header">💳 Payment Management</div>
                            <div class="section-content">
                                <div class="payment-summary">
                                    <h4>Payment Summary</h4>
                                    <div class="payment-row">
                                        <span>Total Amount Due:</span>
                                        <span>₹<?php echo number_format($total_price, 2); ?></span>
                                    </div>
                                    <div class="payment-row">
                                        <span>Total Paid:</span>
                                        <span>₹<?php echo number_format($total_paid, 2); ?></span>
                                    </div>
                                    <div class="payment-row">
                                        <span>Remaining Due:</span>
                                        <span style="color: <?php echo $remaining_due > 0 ? '#ef4444' : '#059669'; ?>; font-weight: 600;">₹<?php echo number_format($remaining_due, 2); ?></span>
                                    </div>
                                </div>

                                <!-- Trigger Button for Payment Modal -->
                              
                        <!-- Payment Modal -->
                        <div class="payment-modal" id="payment-modal">
                            <div class="payment-modal-content">
                                <div class="payment-modal-header">
                                    <span>💳 Record New Payment</span>
                                    <button type="button" class="payment-modal-close" id="close-payment-modal">&times;</button>
                                </div>
                                <div class="payment-modal-body">
                                    <!-- Payment Form -->
                                    <form method="post">
                                        <input type="hidden" name="pat_id" value="<?php echo $patient_id; ?>">
                                        
                                        <div class="form-grid">
                                            <div class="form-group">
                                                <label for="total_amount_due">Total Amount Due (₹)</label>
                                                <input type="number" class="form-control" id="total_amount_due" name="total_amount_due" value="<?php echo $remaining_due; ?>" readonly>
                                            </div>
                                            
                                            <div class="form-group">
                                                <label for="discount_amount">Discount Amount (₹)</label>
                                                <input type="number" class="form-control" id="discount_amount" name="discount_amount" step="0.01" min="0" value="0">
                                            </div>
                                            
                                            <div class="form-group">
                                                <label for="paid_amount">Paid Amount (₹)</label>
                                                <input type="number" class="form-control" id="paid_amount" name="paid_amount" step="0.01" min="0" max="<?php echo $total_price; ?>" required>
                                            </div>
                                            
                                            <div class="form-group">
                                                <label for="payment_method">Payment Method</label>
                                                <select class="form-control" id="payment_method" name="payment_method" required>
                                                    <option value="">Select Payment Method</option>
                                                    <option value="Cash">💰 Cash</option>
                                                    <option value="Credit Card">💳 Credit Card</option>
                                                    <option value="Debit Card">💳 Debit Card</option>
                                                    <option value="Bank Transfer">📱 UPI / 🌐 Online Banking</option>
                                                    <option value="Insurance">🛡️ Insurance</option>
                                                </select>
                                            </div>
                                            
                                            <div class="form-group">
                                                <label for="remarks">Remarks</label>
                                                <textarea class="form-control" id="remarks" name="remarks" rows="3" placeholder="Additional notes about this payment..."></textarea>
                                            </div>
                                        </div>
                                        
                                        <button type="submit" name="add_payment" class="btn btn-primary">Record Payment</button>
                                        <button type="button" class="btn btn-secondary" id="cancel-payment">Cancel</button>
                                    </form>
                                </div>
                            </div>
                        </div>
                        
                        <?php else: ?>
                        <div class="section">
                            <div class="section-content" style="text-align: center; padding: 40px;">
                                <div class="alert alert-danger">
                                    Patient not found. Please select a valid patient.
                                </div>
                            </div>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

            <!-- Footer Start -->
            <?php include('assets/inc/footer.php');?>
        </div>
    </div>

    <!-- Vendor js -->
    <script src="assets/js/vendor.min.js"></script>
    <script src="assets/js/app.min.js"></script>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>

    <script>
        // Add new medicine row
        function addMedicineRow() {
            const medicineRows = document.getElementById('medicine-rows');
            const newRow = document.createElement('div');
            newRow.className = 'medicine-row';
            newRow.innerHTML = `
                <div class="medicine-fields">
                    <div class="form-group">
                        <label>Medicine Name</label>
                        <input type="text" name="medicine_name[]" required>
                    </div>
                    <div class="form-group">
                        <label>Description</label>
                        <input type="text" name="medicine_description[]" required>
                    </div>
                    <div class="form-group">
                        <label>Quantity</label>
                        <input type="number" name="medicine_qty[]" min="1" required>
                    </div>
                    <div class="form-group">
                        <label>Price (₹)</label>
                        <input type="number" name="medicine_price[]" step="0.01" min="0" required>
                    </div>
                </div>
                <button type="button" class="btn btn-danger" onclick="removeMedicineRow(this)">Remove</button>
            `;
            medicineRows.appendChild(newRow);
        }

        // Remove medicine row
        function removeMedicineRow(button) {
            const rows = document.querySelectorAll('.medicine-row');
            if (rows.length > 1) {
                button.closest('.medicine-row').remove();
            } else {
                alert('You need at least one medicine row.');
            }
        }

        // Auto-calculate paid amount based on discount
        document.getElementById('discount_amount').addEventListener('input', function() {
            const totalDue = parseFloat(document.getElementById('total_amount_due').value) || 0;
            const discount = parseFloat(this.value) || 0;
            const maxPayable = totalDue - discount;
            const paidInput = document.getElementById('paid_amount');
            paidInput.max = maxPayable;
            
            if (parseFloat(paidInput.value) > maxPayable) {
                paidInput.value = maxPayable;
            }
        });

        // Validate paid amount doesn't exceed available amount
        document.getElementById('paid_amount').addEventListener('input', function() {
            const totalDue = parseFloat(document.getElementById('total_amount_due').value) || 0;
            const discount = parseFloat(document.getElementById('discount_amount').value) || 0;
            const maxPayable = totalDue - discount;
            
            if (parseFloat(this.value) > maxPayable) {
                this.value = maxPayable;
                alert(`Payment cannot exceed ₹${maxPayable.toFixed(2)} after discount.`);
            }
        });

        // Payment Modal Functionality
        const paymentModal = document.getElementById('payment-modal');
        const openPaymentModalBtn = document.getElementById('open-payment-modal');
        const closePaymentModalBtn = document.getElementById('close-payment-modal');
        const cancelPaymentBtn = document.getElementById('cancel-payment');

        function openPaymentModal() {
            paymentModal.classList.add('active');
            document.body.style.overflow = 'hidden'; // Prevent scrolling
        }

        function closePaymentModal() {
            paymentModal.classList.remove('active');
            document.body.style.overflow = ''; // Re-enable scrolling
        }

        openPaymentModalBtn.addEventListener('click', openPaymentModal);
        closePaymentModalBtn.addEventListener('click', closePaymentModal);
        cancelPaymentBtn.addEventListener('click', closePaymentModal);

        // Close modal when clicking outside the content
        paymentModal.addEventListener('click', function(e) {
            if (e.target === paymentModal) {
                closePaymentModal();
            }
        });

        // Close modal with Escape key
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape' && paymentModal.classList.contains('active')) {
                closePaymentModal();
            }
        });
    </script>
</body>
</html>