
<?php
// Enable error reporting for development (disable in production)
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Set up error logging
$logDir = __DIR__ . '/logs';
$logFile = $logDir . '/error.log';

// Ensure logs directory exists and is writable
if (!is_dir($logDir)) {
    if (!mkdir($logDir, 0755, true)) {
        // Fallback to PHP's default error log if directory creation fails
        ini_set('log_errors', 1);
        ini_set('error_log', 'php://stderr');
        error_log('Failed to create logs directory: ' . $logDir);
    }
}
if (is_dir($logDir) && !is_writable($logDir)) {
    // Fallback if directory is not writable
    ini_set('log_errors', 1);
    ini_set('error_log', 'php://stderr');
    error_log('Logs directory is not writable: ' . $logDir);
} else {
    ini_set('log_errors', 1);
    ini_set('error_log', $logFile);
}

require_once 'core.php';

$valid = ['success' => false, 'messages' => []];

try {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception('Invalid request method. Only POST is allowed.');
    }

    // Validate required POST fields
    $requiredFields = [
        'supplierName' => 'Supplier ID',
        'medicineName' => 'Product ID',
        'numberOfUnits' => 'Unit Stock',
        'unitType' => 'Unit Type',
        'itemsPerUnit' => 'Items per Unit',
        'unitDescription' => 'Unit Description',
        'packPurchaseRate' => 'Pack Purchase Rate',
        'packSellingRate' => 'Pack Selling Rate',
        'packMRP' => 'Pack MRP',
        'expdate' => 'Expiry Date',
        'cgst' => 'CGST',
        'sgst' => 'SGST'
    ];

    foreach ($requiredFields as $field => $fieldName) {
        if (!isset($_POST[$field]) || trim($_POST[$field]) === '') {
            throw new Exception("Missing or empty required field: $fieldName");
        }
    }

    // Fields for stock management
    $supplierId = $_POST['supplierName'];
    $medicineId = $_POST['medicineName'];
    $numberOfUnits = filter_var($_POST['numberOfUnits'], FILTER_VALIDATE_FLOAT);
    $unitType = trim($_POST['unitType']);
    $itemsPerUnit = filter_var($_POST['itemsPerUnit'], FILTER_VALIDATE_FLOAT);
    $unitDescription = filter_var($_POST['unitDescription'], FILTER_VALIDATE_FLOAT);

    // Validate numeric fields
    if ($numberOfUnits === false || $numberOfUnits <= 0) {
        throw new Exception('Unit Stock must be a valid number greater than 0.');
    }
    if ($itemsPerUnit === false || $itemsPerUnit <= 0) {
        throw new Exception('Items per Unit must be a valid number greater than 0.');
    }
    if ($unitDescription === false || $unitDescription <= 0) {
        throw new Exception('Unit Description must be a valid number greater than 0.');
    }

    // Calculate Final Stock
    $quantity = $numberOfUnits * $unitDescription;

    // Pack pricing
    $packPurchaseRate = filter_var($_POST['packPurchaseRate'], FILTER_VALIDATE_FLOAT);
    $packSellingRate = filter_var($_POST['packSellingRate'], FILTER_VALIDATE_FLOAT);
    $packMRP = filter_var($_POST['packMRP'], FILTER_VALIDATE_FLOAT);

    // Validate pricing fields
    if ($packPurchaseRate === false || $packPurchaseRate <= 0) {
        throw new Exception('Pack Purchase Rate must be a valid number greater than 0.');
    }
    if ($packSellingRate === false || $packSellingRate <= 0) {
        throw new Exception('Pack Selling Rate must be a valid number greater than 0.');
    }
    if ($packMRP === false || $packMRP <= 0) {
        throw new Exception('Pack MRP must be a valid number greater than 0.');
    }

    // Per item pricing (calculated as Pack Price ÷ Unit Description)
    $purchaseRate = $unitDescription > 0 ? $packPurchaseRate / $unitDescription : 0;
    $rate = $unitDescription > 0 ? $packSellingRate / $unitDescription : 0;
    $mrp = $unitDescription > 0 ? $packMRP / $unitDescription : 0;

    // Validate per item prices
    if ($purchaseRate <= 0 || $rate <= 0 || $mrp <= 0) {
        throw new Exception('Calculated per item prices must be greater than 0.');
    }

    $bno = !empty($_POST['bno']) ? trim($_POST['bno']) : null;
    $expdate = $_POST['expdate'];
    $cgst = filter_var($_POST['cgst'], FILTER_VALIDATE_FLOAT);
    $sgst = filter_var($_POST['sgst'], FILTER_VALIDATE_FLOAT);
    $chamber = !empty($_POST['chamber']) ? trim($_POST['chamber']) : null;
    $editMode = isset($_POST['editMode']) && $_POST['editMode'] === '1';
    $productId = $editMode ? $_POST['product_id'] : $medicineId;

    // Validate numeric tax fields
    if ($cgst === false || $cgst < 0) {
        throw new Exception('CGST must be a valid number greater than or equal to 0.');
    }
    if ($sgst === false || $sgst < 0) {
        throw new Exception('SGST must be a valid number greater than or equal to 0.');
    }

    // Validate product ID
    if (empty($productId)) {
        throw new Exception('Product ID is missing.');
    }

    // Validate expiry date format
    if (!DateTime::createFromFormat('Y-m-d', $expdate)) {
        throw new Exception('Invalid expiry date format.');
    }

    $orderDate = date('Y-m-d');

    // Log input data for debugging (sanitized)
    error_log(date('Y-m-d H:i:s') . ' [INFO] createStock.php: Processing request - ' .
        'supplierId=' . $supplierId . ', medicineId=' . $medicineId . ', quantity=' . $quantity .
        ', purchaseRate=' . $purchaseRate . ', rate=' . $rate . ', mrp=' . $mrp, 3, $logFile);

    // Use prepared statements for security
    if ($editMode) {
        // Update existing stock
        $sql = "UPDATE product SET 
                    supplier_id = ?, 
                    quantity = ?, 
                    units_per_pack = ?, 
                    unit = ?, 
                    unit_description = ?, 
                    rate = ?, 
                    purchase_rate = ?, 
                    mrp = ?, 
                    pack_purchase_rate = ?, 
                    pack_selling_rate = ?, 
                    pack_mrp = ?, 
                    bno = ?, 
                    expdate = ?, 
                    cgst = ?, 
                    sgst = ?, 
                    chamber = ?, 
                    active = 1 
                WHERE product_id = ?";
    } else {
        // Add new stock (increment quantity)
        $sql = "UPDATE product SET 
                    supplier_id = ?, 
                    quantity = quantity + ?, 
                    units_per_pack = ?, 
                    unit = ?, 
                    unit_description = ?, 
                    rate = ?, 
                    purchase_rate = ?, 
                    mrp = ?, 
                    pack_purchase_rate = ?, 
                    pack_selling_rate = ?, 
                    pack_mrp = ?, 
                    bno = ?, 
                    expdate = ?, 
                    cgst = ?, 
                    sgst = ?, 
                    chamber = ?, 
                    active = 1 
                WHERE product_id = ?";
    }

    $stmt = $connect->prepare($sql);
    if ($stmt === false) {
        throw new Exception('Failed to prepare SQL statement: ' . $connect->error);
    }

    // Bind parameters
    $stmt->bind_param(
        'sddssdssssssssssd',
        $supplierId,
        $quantity,
        $itemsPerUnit,
        $unitType,
        $unitDescription,
        $rate,
        $purchaseRate,
        $mrp,
        $packPurchaseRate,
        $packSellingRate,
        $packMRP,
        $bno,
        $expdate,
        $cgst,
        $sgst,
        $chamber,
        $productId
    );

    // Execute the query
    if ($stmt->execute()) {
        $valid['success'] = true;
        $valid['messages'] = $editMode ? 'Stock Successfully Updated' : 'Stock Successfully Added';
        error_log(date('Y-m-d H:i:s') . ' [SUCCESS] createStock.php: ' . $valid['messages'] . ' for product_id=' . $productId, 3, $logFile);
        header('Location: ../stock_management.php?message=success');
        exit;
    } else {
        throw new Exception('Database query failed: ' . $stmt->error);
    }

} catch (Exception $e) {
    // Log the error
    error_log(date('Y-m-d H:i:s') . ' [ERROR] createStock.php: ' . $e->getMessage(), 3, $logFile);
    
    // Return user-friendly error message
    $valid['success'] = false;
    $valid['messages'] = 'Error: ' . $e->getMessage();
    echo json_encode($valid);
    exit;
} finally {
    if (isset($stmt)) {
        $stmt->close();
    }
    $connect->close();
}
?>
