<?php
header('Content-Type: application/json');
require_once '../constant/connect.php';

error_reporting(E_ALL);
ini_set('display_errors', 1);

$response = ['success' => false, 'messages' => ''];

try {
    // Log POST data for debugging
    error_log("POST data: " . print_r($_POST, true));

    // Validate input
    $order_id = intval($_POST['order_id'] ?? 0);
    $orderDate = mysqli_real_escape_string($connect, $_POST['orderDate'] ?? '');
    $clientName = mysqli_real_escape_string($connect, $_POST['clientName'] ?? '');
    $clientContact = mysqli_real_escape_string($connect, $_POST['clientContact'] ?? '');
    $gstn = mysqli_real_escape_string($connect, $_POST['gstn'] ?? '');
    $subTotal = floatval($_POST['subTotalValue'] ?? 0);
    $totalAmount = floatval($_POST['totalAmountValue'] ?? 0);
    $grandTotal = floatval($_POST['grandTotalValue'] ?? 0);
    $paid = floatval($_POST['paid'] ?? 0);
    $due = floatval($_POST['dueValue'] ?? 0);
    $paymentType = intval($_POST['paymentType'] ?? 0);
    $paymentStatus = intval($_POST['paymentStatus'] ?? 0);
    $type = mysqli_real_escape_string($connect, $_POST['type'] ?? 'retail');
    $productIds = $_POST['productId'] ?? [];
    $quantities = $_POST['quantity'] ?? [];
    $rates = $_POST['rateValue'] ?? [];
    $totals = $_POST['totalValue'] ?? [];
    $discountPercentages = $_POST['discount_percentage_item'] ?? [];
    $productNames = $_POST['productName'] ?? [];

    // Validate required fields
    if (empty($order_id) || empty($orderDate) || empty($clientName) || empty($clientContact) || empty($paymentType) || empty($paymentStatus)) {
        throw new Exception('All required fields must be filled');
    }

    if (empty($productIds) || count($productIds) === 0) {
        throw new Exception('At least one product is required');
    }

    // Start transaction
    $connect->begin_transaction();

    // Update orders table
    $query = "UPDATE orders SET 
        orderDate = ?, 
        clientName = ?, 
        clientContact = ?, 
        gstn = ?, 
        subTotal = ?, 
        totalAmount = ?, 
        grandTotalValue = ?, 
        paid = ?, 
        dueValue = ?, 
        paymentType = ?, 
        paymentStatus = ?, 
        type = ? 
        WHERE id = ?";
    $stmt = $connect->prepare($query);
    if (!$stmt) {
        throw new Exception('Prepare failed for orders update: ' . $connect->error);
    }

    // Bind parameters (sssiiddiisisi)
    $stmt->bind_param(
        "sssiiddiisisi",
        $orderDate,        // s (date)
        $clientName,       // s (text)
        $clientContact,    // s (varchar)
        $gstn,             // s (varchar)
        $subTotal,         // i (decimal, treated as integer for binding)
        $totalAmount,      // i (decimal, treated as integer)
        $grandTotal,       // d (decimal)
        $paid,             // i (decimal, treated as integer)
        $due,              // i (decimal, treated as integer)
        $paymentType,      // i (int)
        $paymentStatus,    // i (int)
        $type,             // s (varchar)
        $order_id          // i (int)
    );
    if (!$stmt->execute()) {
        throw new Exception('Failed to update orders table: ' . $stmt->error);
    }

    // Fetch existing order items to restore stock
    $existingItemsQuery = "SELECT product_id, quantity FROM order_item WHERE order_id = ?";
    $existingStmt = $connect->prepare($existingItemsQuery);
    $existingStmt->bind_param("i", $order_id);
    $existingStmt->execute();
    $existingItems = $existingStmt->get_result()->fetch_all(MYSQLI_ASSOC);

    // Restore stock for existing items
    foreach ($existingItems as $item) {
        $productId = $item['product_id'];
        $oldQuantity = floatval($item['quantity']);
        $updateQuery = "UPDATE product SET quantity = quantity + ? WHERE product_id = ?";
        $updateStmt = $connect->prepare($updateQuery);
        $updateStmt->bind_param("di", $oldQuantity, $productId);
        if (!$updateStmt->execute()) {
            throw new Exception("Failed to restore stock for product ID: $productId: " . $updateStmt->error);
        }
    }

    // Delete existing order items
    $deleteQuery = "DELETE FROM order_item WHERE order_id = ?";
    $deleteStmt = $connect->prepare($deleteQuery);
    $deleteStmt->bind_param("i", $order_id);
    if (!$deleteStmt->execute()) {
        throw new Exception('Failed to delete existing order items: ' . $deleteStmt->error);
    }

    // Insert new order items
    $query = "INSERT INTO order_item (order_id, product_id, productName, quantity, rate, total, discount_percentage, added_date)
              VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
    $stmt = $connect->prepare($query);
    if (!$stmt) {
        throw new Exception('Prepare failed for order_item insert: ' . $connect->error);
    }

    for ($i = 0; $i < count($productIds); $i++) {
        $productId = intval($productIds[$i]);

        // Fetch product name from database
        $productQuery = "SELECT product_name, quantity FROM product WHERE product_id = ? AND status = 1 AND active = 1";
        $productStmt = $connect->prepare($productQuery);
        $productStmt->bind_param("i", $productId);
        $productStmt->execute();
        $productResult = $productStmt->get_result();
        if ($productResult->num_rows === 0) {
            throw new Exception("Invalid product ID: $productId");
        }
        $product = $productResult->fetch_assoc();
        $productName = $product['product_name'] ?: 'Unknown Product';
        $availableQuantity = floatval($product['quantity']);

        $quantity = floatval($quantities[$i]);
        $rate = floatval($rates[$i]);
        $total = floatval($totals[$i]);
        $discountPercentage = floatval($discountPercentages[$i]);
        $addedDate = date('Y-m-d');

        if ($quantity > $availableQuantity) {
            throw new Exception("Insufficient stock for product ID: $productId");
        }

        if (empty($productName) || $productName === 'Unknown Product') {
            throw new Exception("Product name cannot be empty for product ID: $productId");
        }

        // Insert order item
        $stmt->bind_param("iisdddds", $order_id, $productId, $productName, $quantity, $rate, $total, $discountPercentage, $addedDate);
        if (!$stmt->execute()) {
            throw new Exception("Failed to insert order item for product ID: $productId: " . $stmt->error);
        }

        // Update product quantity
        $updateQuery = "UPDATE product SET quantity = quantity - ? WHERE product_id = ?";
        $updateStmt = $connect->prepare($updateQuery);
        $updateStmt->bind_param("di", $quantity, $productId);
        if (!$updateStmt->execute()) {
            throw new Exception("Failed to update stock for product ID: $productId: " . $updateStmt->error);
        }
    }

    // Update or insert into his_ledger
    $ledgerQuery = "SELECT order_id FROM his_ledger WHERE order_id = ?";
    $ledgerStmt = $connect->prepare($ledgerQuery);
    if (!$ledgerStmt) {
        throw new Exception('Prepare failed for his_ledger select: ' . $connect->error);
    }
    $ledgerStmt->bind_param("i", $order_id);
    $ledgerStmt->execute();
    $ledgerResult = $ledgerStmt->get_result();

    if ($ledgerResult->num_rows > 0) {
        // Update existing ledger entry
        $ledgerQuery = "UPDATE his_ledger SET transaction_date = ?, description = 'Invoice Payment', debit = ?, credit = 0.00, balance = ? WHERE order_id = ?";
        $ledgerStmt = $connect->prepare($ledgerQuery);
        if (!$ledgerStmt) {
            throw new Exception('Prepare failed for his_ledger update: ' . $connect->error);
        }
        $ledgerStmt->bind_param("sddi", $orderDate, $grandTotal, $grandTotal, $order_id);
    } else {
        // Insert new ledger entry
        $ledgerQuery = "INSERT INTO his_ledger (order_id, transaction_date, description, debit, credit, balance, created_at)
                        VALUES (?, ?, 'Invoice Payment', ?, 0.00, ?, NOW())";
        $ledgerStmt = $connect->prepare($ledgerQuery);
        if (!$ledgerStmt) {
            throw new Exception('Prepare failed for his_ledger insert: ' . $connect->error);
        }
        $ledgerStmt->bind_param("isdd", $order_id, $orderDate, $grandTotal, $grandTotal);
    }
    if (!$ledgerStmt->execute()) {
        throw new Exception('Failed to update/insert ledger entry: ' . $ledgerStmt->error);
    }

    // Commit transaction
    $connect->commit();

    $response['success'] = true;
    $response['messages'] = 'Invoice updated successfully';
} catch (Exception $e) {
    $connect->rollback();
    $response['messages'] = $e->getMessage();
    error_log("Error in editorder.php: " . $e->getMessage());
    http_response_code(500);
}

echo json_encode($response);
$connect->close();
exit;
?>