<?php
header('Content-Type: application/json');
require_once '../constant/connect.php';

error_reporting(E_ALL);
ini_set('display_errors', 1);

$response = ['success' => false, 'messages' => ''];

try {
    // Log POST data for debugging
    error_log("POST data: " . print_r($_POST, true));

    // Validate input
    $orderDate = $_POST['orderDate'] ?? '';
    $orderTime = $_POST['orderTime'] ?? '';
    $clientName = $_POST['clientName'] ?? '';
    $clientContact = $_POST['clientContact'] ?? '';
    $subTotal = floatval($_POST['subTotalValue'] ?? 0);
    $totalAmount = floatval($_POST['totalAmountValue'] ?? 0);
    $grandTotal = floatval($_POST['grandTotalValue'] ?? 0);
    $paid = floatval($_POST['paid'] ?? 0);
    $due = floatval($_POST['dueValue'] ?? 0);
    $paymentType = intval($_POST['paymentType'] ?? 0);
    $paymentStatus = intval($_POST['paymentStatus'] ?? 0);
    $gstn = $_POST['gstn'] ?? '';
    $uno = $_POST['uno'] ?? '';
    $type = $_POST['type'] ?? 'retail';
    $productIds = $_POST['productId'] ?? [];
    $quantities = $_POST['quantity'] ?? [];
    $rates = $_POST['rateValue'] ?? [];
    $totals = $_POST['totalValue'] ?? [];
    $discountPercentages = $_POST['discount_percentage_item'] ?? [];
    $packs = $_POST['pack'] ?? [];
    $expiryDates = $_POST['expiry_date'] ?? [];
    $realMedicineNames = $_POST['realMedicineName'] ?? [];

    // Validate required fields
    if (empty($orderDate) || empty($orderTime) || empty($clientName) || empty($clientContact) || empty($paymentType) || empty($paymentStatus)) {
        throw new Exception('All required fields must be filled');
    }

    if (empty($productIds) || count($productIds) === 0) {
        throw new Exception('At least one product is required');
    }

    // Start transaction
    $connect->begin_transaction();

    // Insert into orders table
    $query = "INSERT INTO orders (uno, orderDate, clientName, clientContact, address, subTotal, totalAmount, discount, grandTotalValue, gstn, paid, dueValue, paymentType, paymentStatus, delete_status, type)
              VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
    $stmt = $connect->prepare($query);
    if (!$stmt) {
        throw new Exception('Prepare failed: ' . $connect->error);
    }

    $address = ''; // Hardcoded as per schema
    $discount = 0; // Hardcoded as per query
    $delete_status = 0; // Hardcoded as per query

    // Bind parameters (16 total: sssisiiiisiiisis)
    $stmt->bind_param(
        "sssisiiiisiiisis",
        $uno,               // s (varchar)
        $orderDate,        // s (date)
        $clientName,       // s (text)
        $clientContact,    // s (now varchar)
        $address,          // s (varchar)
        $subTotal,         // i (int)
        $totalAmount,      // i (int)
        $discount,         // i (int)
        $grandTotal,       // i (int)
        $gstn,             // s (now varchar)
        $paid,             // i (int)
        $due,              // i (int)
        $paymentType,      // i (int)
        $paymentStatus,    // i (int)
        $delete_status,    // i (tinyint)
        $type              // s (varchar)
    );
    $stmt->execute();
    $orderId = $connect->insert_id;

    // Insert into order_item table with new fields
    $query = "INSERT INTO order_item (productName, quantity, rate, total, order_id, product_id, discount_percentage, added_date, pack, expiry_date, real_medicine_name)
              VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
    $stmt = $connect->prepare($query);
    if (!$stmt) {
        throw new Exception('Prepare failed: ' . $connect->error);
    }

    for ($i = 0; $i < count($productIds); $i++) {
        $productId = intval($productIds[$i]);
        // Fetch product name
        $productQuery = "SELECT product_name, quantity FROM product WHERE product_id = ?";
        $productStmt = $connect->prepare($productQuery);
        $productStmt->bind_param("i", $productId);
        $productStmt->execute();
        $productResult = $productStmt->get_result();
        $product = $productResult->fetch_assoc();
        if (!$product) {
            throw new Exception("Invalid product ID: $productId");
        }
        $productName = $product['product_name'];
        $availableQuantity = floatval($product['quantity']);

        $quantity = floatval($quantities[$i]);
        if ($quantity > $availableQuantity) {
            throw new Exception("Insufficient stock for product ID: $productId");
        }

        $rate = floatval($rates[$i]);
        $total = floatval($totals[$i]);
        $discountPercentage = floatval($discountPercentages[$i]);
        $addedDate = date('Y-m-d');
        $pack = $packs[$i] ?? '';
        $expiryDate = $expiryDates[$i] ?? '';
        $realMedicineName = $realMedicineNames[$i] ?? '';

        $stmt->bind_param("sdddiidssss", $productName, $quantity, $rate, $total, $orderId, $productId, $discountPercentage, $addedDate, $pack, $expiryDate, $realMedicineName);
        $stmt->execute();

        // Update product quantity
        $updateQuery = "UPDATE product SET quantity = quantity - ? WHERE product_id = ?";
        $updateStmt = $connect->prepare($updateQuery);
        $updateStmt->bind_param("di", $quantity, $productId);
        $updateStmt->execute();
        if ($updateStmt->affected_rows === 0) {
            throw new Exception("Failed to update stock for product ID: $productId");
        }
    }

    // Insert into his_ledger
    $ledgerQuery = "INSERT INTO his_ledger (order_id, transaction_date, description, debit, credit, balance, created_at)
                    VALUES (?, ?, 'Invoice Payment', ?, 0.00, ?, NOW())";
    $ledgerStmt = $connect->prepare($ledgerQuery);
    if (!$ledgerStmt) {
        throw new Exception('Prepare failed: ' . $connect->error);
    }
    $ledgerStmt->bind_param("isdd", $orderId, $orderDate, $grandTotal, $grandTotal);
    $ledgerStmt->execute();

    // Commit transaction
    $connect->commit();

    $response['success'] = true;
    $response['messages'] = 'Order created successfully';
} catch (Exception $e) {
    $connect->rollback();
    $response['messages'] = $e->getMessage();
    error_log("Error in order.php: " . $e->getMessage());
}

echo json_encode($response);
$connect->close();
exit;
?>