<?php
// constant/sync_wholesale_ledger.php

function syncWholesaleToLedger($wholesale_order_id) {
    global $connect;
    
    try {
        $connect->autocommit(FALSE); // Start transaction
        
        $orderSql = "SELECT * FROM wholesale_orders WHERE id = ? AND delete_status = 0";
        $orderStmt = $connect->prepare($orderSql);
        $orderStmt->bind_param("i", $wholesale_order_id);
        $orderStmt->execute();
        $order = $orderStmt->get_result()->fetch_assoc();
        
        if (!$order) {
            throw new Exception("Wholesale order not found");
        }
        
        // Check if already synced
        $checkSql = "SELECT * FROM his_ledger WHERE wholesale_order_id = ?";
        $checkStmt = $connect->prepare($checkSql);
        $checkStmt->bind_param("i", $wholesale_order_id);
        $checkStmt->execute();
        $existingEntry = $checkStmt->get_result()->fetch_assoc();
        
        if ($existingEntry) {
            // Calculate difference for balance adjustment
            $amountDiff = $order['grand_total'] - $existingEntry['debit'];
            
            // Update existing entry
            $updateSql = "UPDATE his_ledger SET 
                         debit = ?, 
                         description = ?, 
                         transaction_date = ?,
                         balance = balance + ?
                         WHERE wholesale_order_id = ?";
            $updateStmt = $connect->prepare($updateSql);
            $updateStmt->bind_param("dssdi", 
                $order['grand_total'],
                "Wholesale Invoice: " . $order['invoice_no'],
                $order['invoice_date'],
                $amountDiff,
                $wholesale_order_id
            );
            
            if (!$updateStmt->execute()) {
                throw new Exception("Failed to update ledger entry");
            }
            
            // Update subsequent balances
            $updateSubsequentSql = "UPDATE his_ledger 
                                  SET balance = balance + ? 
                                  WHERE ledger_id > ?";
            $updateSubsequentStmt = $connect->prepare($updateSubsequentSql);
            $updateSubsequentStmt->bind_param("di", $amountDiff, $existingEntry['ledger_id']);
            
            if (!$updateSubsequentStmt->execute()) {
                throw new Exception("Failed to update subsequent balances");
            }
            
        } else {
            // Get last balance
            $lastBalanceSql = "SELECT balance FROM his_ledger ORDER BY ledger_id DESC LIMIT 1";
            $lastBalanceResult = $connect->query($lastBalanceSql);
            $lastBalance = $lastBalanceResult->num_rows > 0 ? $lastBalanceResult->fetch_assoc()['balance'] : 0;
            $newBalance = $lastBalance + $order['grand_total'];
            
            // Insert new entry
            $insertSql = "INSERT INTO his_ledger 
                         (wholesale_order_id, transaction_date, description, debit, credit, balance, transaction_type) 
                         VALUES (?, ?, ?, ?, ?, ?, 'wholesale')";
            $insertStmt = $connect->prepare($insertSql);
            $insertStmt->bind_param("issddds", 
                $wholesale_order_id, 
                $order['invoice_date'], 
                "Wholesale Invoice: " . $order['invoice_no'],
                $order['grand_total'], 
                0, 
                $newBalance
            );
            
            if (!$insertStmt->execute()) {
                throw new Exception("Failed to insert ledger entry");
            }
        }
        
        $connect->commit();
        return true;
        
    } catch (Exception $e) {
        $connect->rollback();
        error_log("Ledger Sync Error: " . $e->getMessage());
        return false;
    } finally {
        $connect->autocommit(TRUE);
    }
}

// Function to sync all wholesale orders (for initial setup or repair)
function syncAllWholesaleToLedger() {
    global $connect;
    
    $ordersSql = "SELECT id FROM wholesale_orders WHERE delete_status = 0 ORDER BY id";
    $ordersResult = $connect->query($ordersSql);
    
    $successCount = 0;
    $errorCount = 0;
    
    while ($order = $ordersResult->fetch_assoc()) {
        if (syncWholesaleToLedger($order['id'])) {
            $successCount++;
        } else {
            $errorCount++;
        }
    }
    
    return array('success' => $successCount, 'errors' => $errorCount);
}

// Function to remove ledger entry when order is deleted
function removeWholesaleFromLedger($wholesale_order_id) {
    global $connect;
    
    try {
        $connect->autocommit(FALSE);
        
        // Get the amount and ledger ID before deletion
        $getEntrySql = "SELECT debit, ledger_id FROM his_ledger WHERE wholesale_order_id = ?";
        $getEntryStmt = $connect->prepare($getEntrySql);
        $getEntryStmt->bind_param("i", $wholesale_order_id);
        $getEntryStmt->execute();
        $entry = $getEntryStmt->get_result()->fetch_assoc();
        
        if (!$entry) {
            return true; // Already not in ledger
        }
        
        // Delete the entry
        $deleteSql = "DELETE FROM his_ledger WHERE wholesale_order_id = ?";
        $deleteStmt = $connect->prepare($deleteSql);
        $deleteStmt->bind_param("i", $wholesale_order_id);
        
        if (!$deleteStmt->execute()) {
            throw new Exception("Failed to delete ledger entry");
        }
        
        // Adjust subsequent balances
        $adjustSql = "UPDATE his_ledger 
                     SET balance = balance - ? 
                     WHERE ledger_id > ?";
        $adjustStmt = $connect->prepare($adjustSql);
        $adjustStmt->bind_param("di", $entry['debit'], $entry['ledger_id']);
        
        if (!$adjustStmt->execute()) {
            throw new Exception("Failed to adjust subsequent balances");
        }
        
        $connect->commit();
        return true;
        
    } catch (Exception $e) {
        $connect->rollback();
        error_log("Ledger Removal Error: " . $e->getMessage());
        return false;
    } finally {
        $connect->autocommit(TRUE);
    }
}
?>